/**
* \file: AilAudioSinkImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
*          D. Girnus  / ADITG/ESM / dgirnus@de.adit-jv.com
*
* \copyright (c) 2016-2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <sys/prctl.h>
#include <adit_logging.h>

#include <AudioFactory.h>
#include <AudioBackend.h>
#include <AudioTypes.h>

#include <inttypes.h>
#include <chrono>

#include "AilAudioSinkImpl.h"

LOG_IMPORT_CONTEXT(aauto_audio)

namespace adit { namespace aauto {

using namespace adit::utility::audio;
using namespace adit::utility;

AilAudioSinkImpl::AilAudioSinkImpl(AudioSink* inSink)
{
    mInitialized = false;
    mStarted = false;
    mSessionId = -1;
    mShutdown = false;

    audioSink = inSink;
    mCallbacks = nullptr;
    mIsSetThreadParam = false;
    mThresholdCounter = 0;

    AudioSinkRecordRunning = false;
    mPlayItemTimeMs = 0;

    setProcessingState(AudioState::STOP);
}

AilAudioSinkImpl::~AilAudioSinkImpl()
{
    if (!mShutdown)
    {
        shutdown();
    }
}

bool AilAudioSinkImpl::init()
{
    if (mInitialized)
    {
        LOG_ERROR((aauto_audio, "%s is already initialized!", sinkName(mConfig.mStreamType)));
        return false;
    }

    /* read and set configuration which was set by Application */
    if (true != getCurrConfig()) {
        LOG_ERROR((aauto_audio, "%s, init()  getCurrConfig() failed", sinkName(mConfig.mStreamType)));
        return false;
    }

    /* load AIL backend */
    std::string backendLibName("Alsa");
    backend = Factory::Instance()->createBackend(backendLibName,*this);

    /* register callbacks to GalReceiver */
    audioSink->registerCallbacks(this);
    /* set GalReceiver::AudioSink configuration */
    audioSink->setMaxUnackedFrames(mConfig.mUnackedFrames);
    audioSink->setCodecType(mConfig.mCodec);
    audioSink->setAudioType(mConfig.mStreamType);
    (void)audioSink->addSupportedConfiguration(mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels);

    mInitialized = true;
    mShutdown = false;
    return true;
}

void AilAudioSinkImpl::shutdown()
{
    mInitialized = false;
    mShutdown = true;

    // in case we where still running stop and notify
    if (mStarted)
    {
        stop(true /* in case of shutdown, discard remaining work items */);
        if (mCallbacks != nullptr)
            mCallbacks->playbackStopCallback(mSessionId);
    }

    if (AudioSinkRecordRunning) {
        mAudioSinkRecord.close();
        AudioSinkRecordRunning = false;
    }

    LOGD_DEBUG((aauto_audio, "%s, id=%d is down", sinkName(mConfig.mStreamType), mSessionId));
}

bool AilAudioSinkImpl::getCurrConfig(void)
{
    // read out configuration parameters
    if (true != mConfig.ResultConfig())
    {
        LOG_ERROR((aauto_audio, "getCurrConfig()  ResultConfig() failed"));
        return false;
    }

    if (mConfig.mConfSType == "AUDIO_STREAM_MEDIA")
    {
        mConfig.mStreamType = AUDIO_STREAM_MEDIA;
    }
    else if (mConfig.mConfSType == "AUDIO_STREAM_SYSTEM_AUDIO")
    {
        mConfig.mStreamType = AUDIO_STREAM_SYSTEM_AUDIO;
    }
    else if (mConfig.mConfSType == "AUDIO_STREAM_GUIDANCE")
    {
        mConfig.mStreamType = AUDIO_STREAM_GUIDANCE;
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, not support stream type.", sinkName(mConfig.mStreamType)));
        return false;
    }

    if (mConfig.mConfCodec == "MEDIA_CODEC_AUDIO_PCM")
    { 
        /* supported */
        mConfig.mCodec = MEDIA_CODEC_AUDIO_PCM;
    }
    else if (mConfig.mConfCodec == "MEDIA_CODEC_AUDIO_AAC_LC_ADTS")
    {
        mConfig.mCodec = MEDIA_CODEC_AUDIO_AAC_LC_ADTS;
        LOG_ERROR((aauto_audio, "%s, not support codec type.", sinkName(mConfig.mStreamType)));
        return false;
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, Unknown codec type.", sinkName(mConfig.mStreamType)));
        return false;
    }

    if ((mConfig.mStreamType == AUDIO_STREAM_MEDIA) && 
        (mConfig.mSampleRate == MEDIA_AUDIO_SAMPLING_RATE) && 
        (mConfig.mNumBits == MEDIA_AUDIO_BITS_PER_SAMPLE) && 
        (mConfig.mNumChannels == MEDIA_AUDIO_CHANNELS))
    {
        /* ok */
    }
    else if ((mConfig.mStreamType == AUDIO_STREAM_SYSTEM_AUDIO) && 
            (mConfig.mSampleRate == SYSTEM_AUDIO_SAMPLING_RATE) && 
            (mConfig.mNumBits == SYSTEM_AUDIO_BITS_PER_SAMPLE) && 
            (mConfig.mNumChannels == SYSTEM_AUDIO_CHANNELS))
    {
        /* ok */
    }
    else if ((mConfig.mStreamType == AUDIO_STREAM_GUIDANCE) && 
            (mConfig.mSampleRate == GUIDANCE_AUDIO_SAMPLING_RATE) && 
            (mConfig.mNumBits == GUIDANCE_AUDIO_BITS_PER_SAMPLE) && 
            (mConfig.mNumChannels == GUIDANCE_AUDIO_CHANNELS))
    {
        /* ok */
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, Error setting parameter sampling-rate, bits-per-sample, channels",
                sinkName(mConfig.mStreamType)));
        return false;
    }

    if (mConfig.mUnackedFrames <= 0)
    {
        LOG_ERROR((aauto_audio, "%s, Error setting parameter audio-sink-max-unacked-frames",
                sinkName(mConfig.mStreamType)));
        return false;
    }

    return true;
}

void AilAudioSinkImpl::setConfigItem(string inKey, string inValue)
{
    LOGD_DEBUG((aauto_audio, "setConfigItem: inKey = %s inValue = %s", inKey.c_str(),inValue.c_str())); 
    mConfig.set(inKey, inValue);
}

void AilAudioSinkImpl::registerCallbacks(IAditAudioSinkCallbacks* inCallbacks)
{
    mCallbacks = inCallbacks;
}

/* IAudioSinkCallbacks */
void AilAudioSinkImpl::dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
        uint8_t* data, size_t len)
{
    LOG_FATAL((aauto_audio, "%s, non-zero copy version of dataAvailableCallback is not supported!",
            sinkName(mConfig.mStreamType)));
}

void AilAudioSinkImpl::dataAvailableCallback(int32_t sessionId, uint64_t timestamp,
            const ::shared_ptr<IoBuffer>& galFrame, uint8_t* data, size_t len)
{
    if (mStarted)
    {
        /* lock access to mAudioItemDeque */
        std::unique_lock<std::mutex> guard(mMutex);
        /* create new element */
        ::shared_ptr<AudioItem> audioItem(new AudioItem(this,sessionId,timestamp,galFrame,data,len));
        /* queue new element */
        mAudioItemDeque.push_back(audioItem);

        if (mConfig.mVerbose)
        {
            LOGD_VERBOSE((aauto_audio, "%s, id=%d has data available  (len=%zu), num of AudioItems queued: %zu, timestamp: %" PRIu64 "",
                    sinkName(mConfig.mStreamType), mSessionId, len, mAudioItemDeque.size(), timestamp ));
        }
        guard.unlock();

        /* check if pre-buffering was set */
        if (mThresholdCounter >= 0)
        {
            /* start streaming if pre-buffering was done */
            if (mThresholdCounter == 0)
            {
                /* start streaming -> processing() called */
                setProcessingState(AudioState::CONTINUE);
                AudioError err = backend->startStream();
                if (err != AudioError::OK)
                {
                    LOG_ERROR((aauto_audio, "%s, id=%d startStream() failed, error=%d",
                                sinkName(mConfig.mStreamType), mSessionId, static_cast<uint32_t>(err)));
                }
                else
                {
                    LOGD_DEBUG((aauto_audio, "%s, id=%d stream started",
                                sinkName(mConfig.mStreamType), mSessionId));
                }
            }
            mThresholdCounter--;
        }

        mConditonVar.notify_one();

        /* record received AudioSink data into WAV file
         * in case Application enables the feature */
        if ((mConfig.mAudioRecord) && (AudioSinkRecordRunning)) {
            int res = mAudioSinkRecord.write(data, (unsigned int)len);
            if (res != (int)len) {
                LOG_ERROR((aauto_audio, "%s write data for AudioSinkRecord failed=%d",
                        sinkName(mConfig.mStreamType), res));
            }
        }
    }
    else
    {
        LOG_WARN((aauto_audio, "%s, id=%d is not ready to process data",
                sinkName(mConfig.mStreamType), mSessionId));
    }
}

int AilAudioSinkImpl::codecConfigCallback(uint8_t* data, size_t len)
{
    /* Invoked when there is new codec configuration data.
     * Currently, only PCM is supported */
    LOGD_DEBUG((aauto_audio, "%s codec config callback", sinkName(mConfig.mStreamType)));
    return STATUS_SUCCESS;
}

int AilAudioSinkImpl::setupCallback(int mediaCodecType)
{
    /* setup each mediaCodecType. like AIL prepare. this AilAudioSink implement was unnecessary. */
    LOGD_DEBUG((aauto_audio, "%s setup callback", sinkName(mConfig.mStreamType)));
    return STATUS_SUCCESS;
}

void AilAudioSinkImpl::playbackStartCallback(int32_t inSessionId)
{
    LOGD_DEBUG((aauto_audio, "%s, playbackStartCallback() id=%d",
            sinkName(mConfig.mStreamType), inSessionId));

    if (mStarted)
    {
        LOG_ERROR((aauto_audio, "%s, id=%d already got playback start!",
                sinkName(mConfig.mStreamType), mSessionId));
        return;
    }

    mStarted = true;
    bool callbackOK = true;
    mSessionId = inSessionId;

    /* null callbacks are OK */
    if (mCallbacks != nullptr)
    {
        /* notify Application to start audio capture */
        mCallbacks->playbackStartCallback(inSessionId);
        callbackOK = true; // TODO add return code to playbackStartCallback
    }

    if (callbackOK)
    {
        /* read and set configuration again because Application
         * got set new configuration values due to playbackStartCallback */
        if (true != getCurrConfig()) {
            LOG_ERROR((aauto_audio, "%s, playbackStartCallback() id=%d getCurrConfig() failed",
                    sinkName(mConfig.mStreamType), mSessionId));

            if (mCallbacks != nullptr)
            {
                mCallbacks->notifyErrorCallback(AUDIO_SINK_CONFIGURATION_ERROR);
            }
        }
        audioSink->setMaxUnackedFrames(mConfig.mUnackedFrames);
        audioSink->setCodecType(mConfig.mCodec);
        (void)audioSink->addSupportedConfiguration(mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels);
        audioSink->setAudioType(mConfig.mStreamType);

        /* start audio playback */
        bool result = start();
        if (result == false)
        {
            if (mCallbacks != nullptr)
            {
                mCallbacks->notifyErrorCallback(AUDIO_SINK_START_ERROR);
            }
        }
    }

    /* open the WAV file to store the AudioSink record */
    if (mConfig.mAudioRecord) {
        if (0 != mAudioSinkRecord.open(mConfig.mAudioRecordFile , sinkName(mConfig.mStreamType))) {
            LOG_ERROR((aauto_audio, "%s, playbackStartCallback() id=%d Open AudioSinkRecord file failed.",
                    sinkName(mConfig.mStreamType), mSessionId));
        } else {
            AudioSinkRecordRunning = true;
        }
    }
}

void AilAudioSinkImpl::playbackStopCallback(int32_t inSessionId)
{
    LOGD_DEBUG((aauto_audio, "%s, playbackStopCallback() id=%d", sinkName(mConfig.mStreamType), inSessionId));

    if (!mStarted)
    {
        LOG_ERROR((aauto_audio, "%s, got stop without start, id=%d!",
                sinkName(mConfig.mStreamType), inSessionId));
        // continue anyway
    }
    if (inSessionId != mSessionId)
    {
        LOG_ERROR((aauto_audio, "%s, id=%d got stop with different id=%d",
                sinkName(mConfig.mStreamType), mSessionId, inSessionId));
        // continue anyway
    }

    /* stop audio playback */
    mStarted = false;
    /* discard remaining work items was configured by Application.
     * By default it's set to false. */
    stop(mConfig.mDiscardFrames);

    /* null callbacks are OK */
    if (mCallbacks != nullptr)
    {
        /* notify Application to stop audio playback */
        mCallbacks->playbackStopCallback(inSessionId);
    }

    mSessionId = -1; // reset session id

    /* insert WAV header and close the AudioSink record*/
    if ((mConfig.mAudioRecord) && (AudioSinkRecordRunning)) {
        if (0 != mAudioSinkRecord.createWavHeader(mConfig.mCodec, mConfig.mNumChannels, mConfig.mNumBits, mConfig.mSampleRate)) {
            LOG_ERROR((aauto_audio, "%s, playbackStopCallback() create wav header for AudioSinkRecord failed.",
                        sinkName(mConfig.mStreamType)));
        }
        mAudioSinkRecord.close();
    }
}

/* private methods */
bool AilAudioSinkImpl::start()
{
    uint32_t samples = 0;

    /* get number of samples in one frame */
    switch (mConfig.mSampleRate) {
        case MEDIA_AUDIO_SAMPLING_RATE:
        {
            samples = SAMPLES_IN_ONE_FRAME_FOR_48KHZ;
            break;
        }
        case GUIDANCE_AUDIO_SAMPLING_RATE:
        // case SYSTEM_AUDIO_SAMPLING_RATE:
        {
            if (GUIDANCE_AUDIO_SAMPLING_RATE != SYSTEM_AUDIO_SAMPLING_RATE) {
                LOG_WARN((aauto_audio, "%s, id=%d Different sample rate for Guidance(%d) and SystemAudio(%d) not handled!",
                        sinkName(mConfig.mStreamType), mSessionId, GUIDANCE_AUDIO_SAMPLING_RATE, SYSTEM_AUDIO_SAMPLING_RATE));
            }
            samples = SAMPLES_IN_ONE_FRAME_FOR_16KHZ;
            break;
        }
        default:
        {
            LOG_ERROR((aauto_audio, "%s, id=%d Unknown sample rate %d. Only 16kHz and 48kHz are supported.",
                    sinkName(mConfig.mStreamType), mSessionId, mConfig.mSampleRate));
            samples = 0;
            break;
        }
    };

    /* calculate playtime [ms] within one frame */
    mPlayItemTimeMs = samples/(mConfig.mSampleRate/1000);

    /* calculate pre-buffering if set */
    if (mConfig.mThresholdMs > 0) {
        /* "+ palyTimeMs -1" is to round to the up. */
        mThresholdCounter  = (mConfig.mThresholdMs + mPlayItemTimeMs -1)/mPlayItemTimeMs;
    } else {
        mThresholdCounter = 0;
        LOGD_DEBUG((aauto_audio, "%s, id=%d No threshold defined. No pre-buffering.", sinkName(mConfig.mStreamType), mSessionId));
    }
    /* Size of one period for jitter buffer(converted to number of samples). */
    uint32_t periodSamples = mConfig.mPeriodms*(mConfig.mSampleRate/1000);

    /* prepare AIL */
    std::string capture("");
    if (!mConfig.mDisablePrio)
    {
        backend->setThreadSched(SCHED_FIFO, mConfig.mThreadPrio);
    }
    if (mConfig.mInitToutms > 0)
    {
        backend->setInitialTimeout(mConfig.mInitToutms);
    }
    backend->setFadeTime( FadeMode::OUT,  StreamDirection::OUT, 0);

    /* open AIL - streaming not started */
    AudioError err = backend->openStream(capture, mConfig.mDevice, AudioFormat::S16_LE, mConfig.mSampleRate, mConfig.mNumChannels, periodSamples);
    if (err != AudioError::OK)
    {
        LOG_ERROR((aauto_audio, "%s, id=%d openStream() failed, error=%d",
                sinkName(mConfig.mStreamType), mSessionId, static_cast<uint32_t>(err)));
        return false;
    }
    else
    {
        LOGD_DEBUG((aauto_audio, "%s, id=%d stream opened", sinkName(mConfig.mStreamType), mSessionId));
    }

    return true;
}

void AilAudioSinkImpl::stop(bool discardRemaining)
{
    if (mShutdown) {
        /* abort streaming */
        setProcessingState(AudioState::ABORT);
        /* notify to wake up processing() */
        mConditonVar.notify_one();
        AudioError err = backend->abortStream();
        if (err != AudioError::OK) {
            LOG_ERROR((aauto_audio, "%s, id=%d abortStream() failed, error=%d",
                    sinkName(mConfig.mStreamType), mSessionId, static_cast<uint32_t>(err)));
        } else {
            LOGD_DEBUG((aauto_audio, "%s, id=%d stream aborted", sinkName(mConfig.mStreamType), mSessionId));
        }
    } else if (discardRemaining) {
        /* stop streaming */
        setProcessingState(AudioState::STOP);
        /* notify to wake up processing() */
        mConditonVar.notify_one();
        AudioError err = backend->stopStream();
        if (err != AudioError::OK) {
            LOG_ERROR((aauto_audio, "%s, id=%d stopStream() failed, error=%d",
                    sinkName(mConfig.mStreamType), mSessionId, static_cast<uint32_t>(err)));
        } else {
            LOGD_DEBUG((aauto_audio, "%s, id=%d stream stopped", sinkName(mConfig.mStreamType), mSessionId));
        }
    } else {
        /* lock access to mAudioItemDeque */
        std::unique_lock<std::mutex> guard(mMutex);
        if ( (mAudioItemDeque.size() > 0) && (getProcessingState() == AudioState::CONTINUE) ){
            LOGD_DEBUG((aauto_audio, "%s, id=%d stop stream on condition variable", sinkName(mConfig.mStreamType), mSessionId));
            mConditonVar.wait(guard);
        }
        guard.unlock();

        /* abort streaming */
        setProcessingState(AudioState::ABORT);
        AudioError err = backend->abortStream();
        if (err != AudioError::OK) {
            LOG_ERROR((aauto_audio, "%s, id=%d abortStream() failed, error=%d",
                    sinkName(mConfig.mStreamType), mSessionId, static_cast<uint32_t>(err)));
        } else {
            LOGD_DEBUG((aauto_audio, "%s, id=%d stream aborted", sinkName(mConfig.mStreamType), mSessionId));
        }
    }

    /* close AIL */
    AudioError err = backend->closeStream();
    if (err != AudioError::OK) {
        LOG_ERROR((aauto_audio, "%s, id=%d closeStream() failed, error=%d",
                sinkName(mConfig.mStreamType), mSessionId, static_cast<uint32_t>(err)));
    } else {
        LOGD_DEBUG((aauto_audio, "%s, id=%d stream closed", sinkName(mConfig.mStreamType), mSessionId));
    }

    /* lock access to mAudioItemDeque */
    std::unique_lock<std::mutex> guard(mMutex);
    LOGD_DEBUG((aauto_audio, "%s, id=% d stop done, num of AudioItem discarded: %zu",
                sinkName(mConfig.mStreamType), mSessionId, mAudioItemDeque.size()));

    /* discard queued AudioItems - if available */
    mAudioItemDeque.clear();
    guard.unlock();
}

/* processing() called by AIL */
AudioState AilAudioSinkImpl::processing(unsigned char *in, unsigned char **out, uint32_t &frames)
{
    frames = 0;
    /* lock mutex to protect access to mAudioItemDeque */
    std::unique_lock<std::mutex> guard(mMutex);

    if (!mAudioItemDeque.empty())
    {
        /* handle if the first element marked as played */
        if (mAudioItemDeque.front()->mWasPlayed)
        {
            /* remove first element from queue */
            mAudioItemDeque.pop_front();
            /* sent ACK to GalReceiver */
            audioSink->ackFrames(mSessionId, 1);
        }
    }

    /* get current processing state */
    AudioState state = getProcessingState();
    /* handle current processing state */
    switch (state)
    {
        case AudioState::CONTINUE:
        {
            /* check if we have audio to process */
            if (mAudioItemDeque.empty())
            {
                mConditonVar.notify_one();
                LOGD_DEBUG((aauto_audio, "%s, id=%d Last element is drained - %s",
                            sinkName(mConfig.mStreamType), mSessionId, (mStarted==true)?("wait for new audio data"):("stop/abort processing") ));
                /* check if we were stopped */
                if (mStarted) {
                    /* sometimes the data comes from GAL a few ms later.
                     * todo talk to multimedia about the 8 ms extra, avoid the magic number */
                    mConditonVar.wait_for(guard, std::chrono::milliseconds(mPlayItemTimeMs + 8));

                    /* get possible state change while we slept */
                    state = getProcessingState();
                } else {
                    /* stop playback was triggered.
                     * notify AIL to stop processing. */
                    state = AudioState::ABORT;
                }
            }
            /* play the first element in the queue */
            if ( (!mAudioItemDeque.empty()) && (state == AudioState::CONTINUE) )
            {
                /* get first element from queue */
                ::shared_ptr<AudioItem> work = mAudioItemDeque.front();
                /* provide audio data to AIL */
                *out = work->mDataPtr;
                frames = work->mLen / ((mConfig.mNumBits / 8) * mConfig.mNumChannels);
                /* mark element as played */
                work->mWasPlayed = true;
            }
            break;
        }
        case AudioState::STOP:
        case AudioState::ABORT:
        {
            LOGD_DEBUG((aauto_audio, "%s, id=%d %s processing, state=%d",
                    sinkName(mConfig.mStreamType), mSessionId, (state==AudioState::STOP)?("Stop"):("Abort"), (int32_t)state));
            break;
        }
        default:
        {
            LOG_WARN((aauto_audio, "%s, id=%d Unknown AudioState %d",
                     sinkName(mConfig.mStreamType), mSessionId, (int32_t)state));
            break;
        }
    }

    if (mConfig.mVerbose)
    {
        LOGD_VERBOSE((aauto_audio, "%s, id=%d processing done, state=%d (AudioItems queued: %zu)",
                sinkName(mConfig.mStreamType), mSessionId, (int32_t)state, mAudioItemDeque.size()));
    }
    return state;
}

/* AIL logging */
void AilAudioSinkImpl::error(const std::string& data) const
{
    LOG_ERROR((aauto_audio, "%s, %s", sinkName(mConfig.mStreamType), data.c_str()));
}
void AilAudioSinkImpl::warning(const std::string& data) const
{
    LOG_WARN((aauto_audio, "%s, %s", sinkName(mConfig.mStreamType), data.c_str()));
}
void AilAudioSinkImpl::info(const std::string& data) const
{
    LOG_INFO((aauto_audio, "%s, %s", sinkName(mConfig.mStreamType), data.c_str()));
}
void AilAudioSinkImpl::debug(const std::string& data) const
{
    LOGD_DEBUG((aauto_audio, "%s, %s", sinkName(mConfig.mStreamType), data.c_str() ));
}
eLogLevel AilAudioSinkImpl::checkLogLevel() const
{
    return eLogLevel::LL_MAX;
}

void AilAudioSinkImpl::statistics(const StreamStatistics &status)
{
    LOGD_DEBUG((aauto_audio, "%s, %s() time:%" PRIu64 ", flag:%d, xrun count:%u",
              sinkName(mConfig.mStreamType), __func__, status.time, status.flag, status.xruns.playback));
}

void AilAudioSinkImpl::eostreaming(const AudioError error)
{
    if (error != AudioError::OK)
    {
        LOG_ERROR((aauto_audio, "%s, %s() Streaming has stopped unexpectedly: %d",
                   sinkName(mConfig.mStreamType), __func__, (int32_t)error));

        /* get current state to notify start error or streaming error */
        aautoErrorCodes notifyError = AUDIO_SINK_START_ERROR;
        if (getProcessingState() == AudioState::CONTINUE) {
            notifyError = AUDIO_SINK_WRITE_ERROR;
        }
        /* streaming was aborted by AIL. Set internal state */
        setProcessingState(AudioState::ABORT);

        /* notify Application about error and expect that Application stops streaming */
        if (mCallbacks != nullptr) {
            mCallbacks->notifyErrorCallback(notifyError);
        }
    }
}

} } /* namespace adit { namespace aauto */
